/**
  Copyright (c) 2010 Freescale Semiconductor
  
  \file  	  DCU.c
  \brief	  This is the DCU Driver File
  \brief	  Configures Freq. and Activates DCU and layers on MPC56xxS
  \author	  Freescale Semiconductor
  \author	  Automotive Systems Solutions Engineering
  \author	  IM, b06623 - After SMcA
  \version	  2.0
  \revision	  $Revision: 275 $
  \date  	  $Date: 2011-11-16 15:44:23 -0600 (Wed, 16 Nov 2011) $
  
  * History:  10/July/2008 - Initial Version (IM)
              13/May/2009 - MISRA Check (IM)
              March-2010  -	Support to Sharp LQ070 (IM)
              April-2010  - Initial Upgrade to formally support DCU3 (IM)
              May-2010	  - Double context support (DCU3 and DCUlite)  
              21Sept2010  - DCU and DCULITE can have different displays
              2March2010  - Added preciser divider for each display

  * MISRA VIOLATIONS:
	- [ MISRA 11.2 ]
	- [ MISRA 16.9 ]  

* Copyright (c) 2010, Freescale, Inc.  All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
  
*/

#include	"DCU_drv.h"
#include    "DCUcommon.h"


volatile struct DCUx_tag * DCU_PTR = &DCU;
uint8_t	DCU_MaxLayersCTX = DCU_DCUMAXLAYERS;

// temp vars for profiling OK to remove if you don't use them
extern uint32_t timing[];
extern uint8_t DMAinc;

/* For DCU */
DCU_CallbackType	DCU_CallbackVSYNC_1;
DCU_CallbackType	DCU_CallbackVSBLANK_1;
DCU_CallbackType	DCU_CallbackLSBFVS_1;
DCU_CallbackType	DCU_CallbackPROGD_1;
DCU_CallbackType	DCU_CallbackDMAD_1;
DCU_Timing_t		DCU_TimingStatus_1;
#ifdef DCULITE
/* For DCULITE */
DCU_CallbackType	DCU_CallbackVSYNC_2;
DCU_CallbackType	DCU_CallbackVSBLANK_2;
DCU_CallbackType	DCU_CallbackLSBFVS_2;
DCU_CallbackType	DCU_CallbackPROGD_2;
DCU_CallbackType	DCU_CallbackDMAD_2;  
DCU_Timing_t		DCU_TimingStatus_2;
#endif

/* init out of reset */
DCU_CallbackType*	DCU_CallbackVSYNC_p = &DCU_CallbackVSYNC_1;
DCU_CallbackType*	DCU_CallbackVSBLANK_p = &DCU_CallbackVSBLANK_1;
DCU_CallbackType*	DCU_CallbackLSBFVS_p = &DCU_CallbackLSBFVS_1; 
DCU_CallbackType*	DCU_CallbackPROGD_p = &DCU_CallbackPROGD_1;
DCU_CallbackType*	DCU_CallbackDMAD_p = &DCU_CallbackDMAD_1;
DCU_Timing_t*		DCU_TimingStatus_p = &DCU_TimingStatus_1;

/* for simplicity */
#define DCU_CallbackVSYNC 	(*DCU_CallbackVSYNC_p)
#define DCU_CallbackVSBLANK	(*DCU_CallbackVSBLANK_p)
#define DCU_CallbackLSBFVS 	(*DCU_CallbackLSBFVS_p)
#define DCU_CallbackPROGD	(*DCU_CallbackPROGD_p)
#define DCU_CallbackDMAD	(*DCU_CallbackDMAD_p)
#define DCU_TimingStatus	(*DCU_TimingStatus_p)

void DCU0_init(uint32_t freq, uint8_t mode)
{   
	uint32_t i;
	
    DCUCTX.BGND.R = 0x00000000;
    DCUCTX.DCU_MODE.B.RASTER_EN=1;
	
	LQ050_setup(0,freq);
	//JDI_GCX156AKM_setup(0,freq);

	/* initialise layers to be off */
	for(i=0;i<32;i++){
		DCUCTX.LAYER[i].CTRLDESCL1.R = 0;  /*  */
		DCUCTX.LAYER[i].CTRLDESCL2.R = 0;  /*  */
		DCUCTX.LAYER[i].CTRLDESCL3.R = 0;  /*  */
		DCUCTX.LAYER[i].CTRLDESCL4.R = 0;  /*  */
		DCUCTX.LAYER[i].CTRLDESCL5.R = 0;  /*  */
		DCUCTX.LAYER[i].CTRLDESCL6.R = 0;  /*  */
		DCUCTX.LAYER[i].CTRLDESCL7.R = 0;  /*  */
		DCUCTX.LAYER[i].CTRLDESCL8.R = 0;  /*  */
		DCUCTX.LAYER[i].CTRLDESCL9.R = 0;  /*  */
		DCU_0.R[i].CTRLDESCL_10.R = 0;  /*  */
		DCU_0.R[i].CTRLDESCL_11.R = 0;  /*  */	
	}

	if (mode > 0)
	{
	DCUCTX.DCU_MODE.B.DCU_MODE = mode;
	DCUCTX.UPDATE_MODE.B.READREG = 1;  /* Initiate manual refresh */
	}	
	
	DCU_drvInit();
	
	/* Wait until frame complete then go to automatic mode */
    while(DCUCTX.UPDATE_MODE.B.READREG) {}
    DCUCTX.UPDATE_MODE.B.MODE = 1;
	
	//enable interrupts
	
}

void DCU1_init(uint32_t freq, uint8_t mode)
{   

}



void DCU_drvInit(void)
{
  DCU_PTR = &DCU;
  /* init out of reset */
  DCU_CallbackVSYNC_p = &DCU_CallbackVSYNC_1;
  DCU_CallbackVSBLANK_p = &DCU_CallbackVSBLANK_1;
  DCU_CallbackLSBFVS_p = &DCU_CallbackLSBFVS_1; 
  DCU_CallbackPROGD_p = &DCU_CallbackPROGD_1;
  DCU_CallbackDMAD_p = &DCU_CallbackDMAD_1;
  DCU_TimingStatus_p = &DCU_TimingStatus_1;
  
  DCUCTX.INT_MASK.R = 0xFFFFFFFF;
  //TODO: remove this is an ugly hack
  DCUCTX.INT_MASK.B.M_LYR_TRANS_FINISH = 0u;  
  
  /* VIOLATION TO THE FOLLOWING RULES  [ MISRA 11.2 ] [ MISRA 16.9 ] */
  /* Rationale: It is required to assign a values to this function that will be used as callback */
  /* the null pointer value indicates the function has not been initialized */
  DCU_CallbackVSYNC	= NULL_PTR;
  DCU_CallbackVSBLANK	= NULL_PTR;
  DCU_CallbackLSBFVS	= NULL_PTR;
  DCU_CallbackPROGD	= NULL_PTR;
  DCU_CallbackDMAD	= NULL_PTR;
  DCU_TimingStatus	= DCU_UNKNOWN;
  
  /* Gamma adjust disable */
  DCUCTX.DCU_MODE.B.EN_GAMMA  = 0u;
  /* Set lines before Vertical sync value */
  DCUCTX.THRESHOLD.B.LS_BF_VS = 50u;
  DCUCTX.DCU_MODE.B.BLEND_ITER = 4; 		
  DCUCTX.DCU_MODE.B.RASTER_EN = 1;
}




void DCU_SelectDCU(uint8_t device)
{
	if(device == DCU_DCU)
	{ 
		DCU_PTR = &DCU;
		DCU_CallbackVSYNC_p = &DCU_CallbackVSYNC_1;
		DCU_CallbackVSBLANK_p = &DCU_CallbackVSBLANK_1;
		DCU_CallbackLSBFVS_p = &DCU_CallbackLSBFVS_1; 
		DCU_CallbackPROGD_p = &DCU_CallbackPROGD_1;
		DCU_TimingStatus_p = &DCU_TimingStatus_1;
		DCU_MaxLayersCTX = DCU_DCUMAXLAYERS;		
	}
#ifdef DCULITE	
	else
	{
		DCU_PTR = (volatile struct DCU_tag	*)(&DCULITE);
		DCU_CallbackVSYNC_p = &DCU_CallbackVSYNC_2;
		DCU_CallbackVSBLANK_p = &DCU_CallbackVSBLANK_2;
		DCU_CallbackLSBFVS_p = &DCU_CallbackLSBFVS_2; 
		DCU_CallbackPROGD_p = &DCU_CallbackPROGD_2;
		DCU_TimingStatus_p = &DCU_TimingStatus_2;
		DCU_CallbackDMAD_p = &DCU_CallbackDMAD_2;
		DCU_MaxLayersCTX = DCU_DCULITEMAXLAYERS;			
	}
#endif		
}


uint8_t DCU_GetCurrentDCU(void)
{
#ifdef DCULITE
	if(DCU_MaxLayersCTX == DCU_DCULITEMAXLAYERS)
		return DCU_DCULITE;
	else
#endif	
		return	DCU_DCU;	
}
/**
* \brief	DCU_SetChroma - Sets the low and high level colors when using chroma keying
* \author	IM, b06623
* \param	uint8_t layer, Configured layer.
* \param	uint32_t max, High boundary for chroma keying.
* \param	uint32_t min, Low boundary for chroma keying.
* \return	void
* \todo
*/
void DCU_SetChroma(uint8_t layer, uint32_t max, uint32_t min)
{
	/* Layer param checking */
    if(layer >= DCU_MaxLayersCTX)
    {
    	return;
    }
    /* Chroma setting */
	DCUCTX.LAYER[ layer ].CTRLDESCL5.B.CKMAX_R		= (uint8_t)((0x00FF0000u & max)>>16u);				
	DCUCTX.LAYER[ layer ].CTRLDESCL6.B.CKMIN_R		= (uint8_t)((0x00FF0000u & min)>>16u);				
	DCUCTX.LAYER[ layer ].CTRLDESCL5.B.CKMAX_G		= (uint8_t)((0x0000FF00u & max)>>8u);				
	DCUCTX.LAYER[ layer ].CTRLDESCL6.B.CKMIN_G		= (uint8_t)((0x0000FF00u & min)>>8u);				
	DCUCTX.LAYER[ layer ].CTRLDESCL5.B.CKMAX_B		= (uint8_t)(0x000000FFu & max);		
	DCUCTX.LAYER[ layer ].CTRLDESCL6.B.CKMIN_B		= (uint8_t)(0x000000FFu & min);  
}


void DCU_SetCallbackVSBLANK( DCU_CallbackType Callback )
{
    if(Callback != NULL_PTR)
    {
		/* VIOLATION TO THE FOLLOWING RULE */
		/* [ MISRA 16.9 ] A function identifier shall only be used with either a preceding &, */
		/* or with a parenthesised parameter list, which may be empty. */

		/* Rationale: It is required to assign a values to this function that will be used as callback */
		DCU_CallbackVSBLANK = Callback;
		/* Enable/disable Vertical blanking interrupt */
		DCUCTX.INT_MASK.B.M_VS_BLANK = 0u;
    }
    else
    {
		DCUCTX.INT_MASK.B.M_VS_BLANK = 1u;
    }
}

/**
* \brief	DCU_SetCallbackLSBFVS - Configures the LSBFVS callback interrupt
* \author	IM, b06623
* \param	DCU_CallbackType Callback, Callback used; NULL_PTR to turn it off.
* \return	void
* \todo
*/
void DCU_SetCallbackLSBFVS( DCU_CallbackType Callback )
{

    if(Callback != NULL_PTR)
    {
		/* VIOLATION TO THE FOLLOWING RULE */
		/* [ MISRA 16.9 ] A function identifier shall only be used with either a preceding &, */
		/* or with a parenthesised parameter list, which may be empty. */

		/* Rationale: It is required to assign a values to this function that will be used as callback */
		DCU_CallbackLSBFVS = Callback;
		/* Enable/disable lines before vertical sync interrupt */
		DCUCTX.INT_MASK.B.M_LS_BF_VS  = 0u;
    }
    else
    {
		DCUCTX.INT_MASK.B.M_LS_BF_VS  = 1u;
    }
}

/**
* \brief	DCU_SetCallbackVSYNC - Configures the VSYNC callback interrupt
* \author	IM, b06623
* \param	DCU_CallbackType Callback, Callback used; NULL_PTR to turn it off.
* \return	void
* \todo
*/
void DCU_SetCallbackVSYNC( DCU_CallbackType Callback )
{   
    if(Callback != NULL_PTR)
    {
		/* VIOLATION TO THE FOLLOWING RULE */
		/* [ MISRA 16.9 ] A function identifier shall only be used with either a preceding &, */
		/* or with a parenthesised parameter list, which may be empty. */

		/* Rationale: It is required to assign a values to this function that will be used as callback */
		DCU_CallbackVSYNC = Callback;
		/* Enable/disable Vertical blanking interrupt */
		DCUCTX.INT_MASK.B.M_VSYNC = 0u;
    }
    else
    {
		DCUCTX.INT_MASK.B.M_VSYNC = 1u;
    }
}

/**
* \brief	DCU_SetCallbackProgDone - Configures the VSYNC callback interrupt
* \author	IM, b06623
* \param	DCU_CallbackType Callback, Callback used; NULL_PTR to turn it off.
* \return	void
* \todo
*/
void DCU_SetCallbackProgDone( DCU_CallbackType Callback )
{   
	if(Callback != NULL_PTR)
    {
		/* VIOLATION TO THE FOLLOWING RULE */
		/* [ MISRA 16.9 ] A function identifier shall only be used with either a preceding &, */
		/* or with a parenthesised parameter list, which may be empty. */

		/* Rationale: It is required to assign a values to this function that will be used as callback */
		DCU_CallbackPROGD = Callback;
		/* Enable/disable Vertical blanking interrupt */
		DCUCTX.INT_MASK.B.M_PROG_END = 0u;
    }
    else
    {
		DCUCTX.INT_MASK.B.M_PROG_END = 1u;
    }
}

/**
* \brief	DCU_SetCallbackDMADone - Configures the DMADONE callback interrupt
* \author	IM, b06623
* \param	DCU_CallbackType Callback, Callback used; NULL_PTR to turn it off.
* \return	void
* \todo
*/
void DCU_SetCallbackDMADone( DCU_CallbackType Callback )
{   
	if(Callback != NULL_PTR)
    {
		/* VIOLATION TO THE FOLLOWING RULE */
		/* [ MISRA 16.9 ] A function identifier shall only be used with either a preceding &, */
		/* or with a parenthesised parameter list, which may be empty. */

		/* Rationale: It is required to assign a values to this function that will be used as callback */
		DCU_CallbackDMAD = Callback;
		/* Enable/disable Vertical blanking interrupt */
#ifdef DCU3		
		DCUCTX.INT_MASK.B.M_DMA_TRANS_FINISH  = 0u;
    }
    else
    {
		DCUCTX.INT_MASK.B.M_DMA_TRANS_FINISH  = 1u;
    }
#else
		DCUCTX.INT_MASK.B.M_DMA_TRANS_FINISH  = 0u;
    }
    else
    {
		DCUCTX.INT_MASK.B.M_DMA_TRANS_FINISH  = 1u;
    }
#endif    
}

/* DCU VS_BLANK, LS_BF_VS,VSYNC */
void DCU_TIMING_ISR( void  )
{
	//PIT.CH[5].TCTRL.B.TEN = 0;			// stop the timer ch 
	//PIT.CH[5].TCTRL.B.TEN = 1;			// start the timer ch, starts a down count from 0xFFFFFFFF  

    if( DCUCTX.INT_STATUS.B.LS_BF_VS != 0u )
    {
		//timing[3] = (*(uint32_t *)(PIT_CVAL2));
                //reg32_write(GPIOA_PTOR, 0x400000);
                DCU_TimingStatus_1 = DCU_LSBFVS;
		/* VIOLATION TO THE FOLLOWING RULE */
		/* [ MISRA 16.9 ] A function identifier shall only be used with either a preceding &, */
		/* or with a parenthesised parameter list, which may be empty. */

		/* Rationale: It is required to assign a values to this function that will be used as callback */
		if (DCU_CallbackLSBFVS_1!= NULL_PTR)
		{
	    	DCU_CallbackLSBFVS_1( );
		}
		DCUCTX.INT_STATUS.R = 0x000004u;
    }
    if( DCUCTX.INT_STATUS.B.VS_BLANK != 0u )
    {
                //timing[4] = (*(uint32_t *)(PIT_CVAL2));
                //reg32_write(GPIOA_PTOR, 0x400000);
                DCU_TimingStatus_1 = DCU_VBLANK;
		/* VIOLATION TO THE FOLLOWING RULE */
		/* [ MISRA 16.9 ] A function identifier shall only be used with either a preceding &, */
		/* or with a parenthesised parameter list, which may be empty. */

		/* Rationale: It is required to assign a value to this function that will be used as callback */
		if (DCU_CallbackVSBLANK_1 != NULL_PTR)
		{
		    DCU_CallbackVSBLANK_1( );
		}
		DCUCTX.INT_STATUS.R = 0x000008u;
    }
    if( DCUCTX.INT_STATUS.B.VSYNC != 0u )
    {
                //timing[5] = (*(uint32_t *)(PIT_CVAL2));
		//reg32_write(GPIOA_PTOR, 0x400000);
                DCU_TimingStatus_1 = DCU_VSYNC;
		/* VIOLATION TO THE FOLLOWING RULE */
		/* [ MISRA 16.9 ] A function identifier shall only be used with either a preceding &, */
		/* or with a parenthesised parameter list, which may be empty. */

		/* Rationale: It is required to assign a value to this function that will be used as callback */
		if (DCU_CallbackVSYNC_1 != NULL_PTR)
		{
		    DCU_CallbackVSYNC_1( );
		}
		DCUCTX.INT_STATUS.R = 0x000001u;
    }
    
    if( DCUCTX.INT_STATUS.B.PROG_END != 0u )
    {   
    	/* Rationale: It is required to assign a values to this function that will be used as callback */
                //timing[6] = (*(uint32_t *)(PIT_CVAL2));
		//reg32_write(GPIOA_PTOR, 0x400000);
                if (DCU_CallbackPROGD_1 != NULL_PTR)
		{
	   		DCU_CallbackPROGD_1( );
		}
		DCUCTX.INT_STATUS.R = 0x00000400u;
    }
    if( DCUCTX.INT_STATUS.B.DMA_TRANS_FINISH != 0u )
    {   
    	/* Rationale: It is required to assign a values to this function that will be used as callback */
                //timing[7] = (*(uint32_t *)(PIT_CVAL2));
		//reg32_write(GPIOA_PTOR, 0x400000);
                if (DCU_CallbackDMAD_1 != NULL_PTR)
		{
	   		DCU_CallbackDMAD_1( );
		}
		DCUCTX.INT_STATUS.R = 0x00004000u;
    }    
    if( DCUCTX.INT_STATUS.B.LYR_TRANS_FINISH != 0u )
    {   
    	/* Rationale: It is required to assign a values to this function that will be used as callback */
                //timing[8] = (*(uint32_t *)(PIT_CVAL2));
		//reg32_write(GPIOA_PTOR, 0x400000);
                /*if (DCU_CallbackDMAD_1 != NULL_PTR)
		{
	   		DCU_CallbackDMAD_1( );
		}*/
		DCUCTX.INT_STATUS.R = 0x00001000u;
    }   

}

/* DCULITE VS_BLANK, LS_BF_VS,VSYNC */
#ifdef DCULITE
void DCULITE_TIMING_ISR(void)
{
    if( DCULITE.INT_STATUS.B.LS_BF_VS != 0u )
    {
		DCU_TimingStatus_2 = DCU_LSBFVS;
		/* VIOLATION TO THE FOLLOWING RULE */
		/* [ MISRA 16.9 ] A function identifier shall only be used with either a preceding &, */
		/* or with a parenthesised parameter list, which may be empty. */

		/* Rationale: It is required to assign a values to this function that will be used as callback */
		if (DCU_CallbackLSBFVS_2!= NULL_PTR)
		{
		    DCU_CallbackLSBFVS_2( );
		}
		DCULITE.INT_STATUS.R = 0x000004u;
    }
    if( DCULITE.INT_STATUS.B.VS_BLANK != 0u )
    {
		DCU_TimingStatus_2 = DCU_VBLANK;
		/* VIOLATION TO THE FOLLOWING RULE */
		/* [ MISRA 16.9 ] A function identifier shall only be used with either a preceding &, */
		/* or with a parenthesised parameter list, which may be empty. */

		/* Rationale: It is required to assign a values to this function that will be used as callback */
		if (DCU_CallbackVSBLANK_2 != NULL_PTR)
		{
		    DCU_CallbackVSBLANK_2( );
		}
		DCULITE.INT_STATUS.R = 0x000008u;
    }
    if( DCULITE.INT_STATUS.B.VSYNC != 0u )
    {
		DCU_TimingStatus_2 = DCU_VSYNC;
		/* VIOLATION TO THE FOLLOWING RULE */
		/* [ MISRA 16.9 ] A function identifier shall only be used with either a preceding &, */
		/* or with a parenthesised parameter list, which may be empty. */

		/* Rationale: It is required to assign a values to this function that will be used as callback */
		if (DCU_CallbackVSYNC_2 != NULL_PTR)
		{
		    DCU_CallbackVSYNC_2( );
		}
		DCULITE.INT_STATUS.R = 0x000001u;
    }
    
    if( DCULITE.INT_STATUS.B.PROG_END != 0u )
    {   
    	/* Rationale: It is required to assign a values to this function that will be used as callback */
		if (DCU_CallbackPROGD_2 != NULL_PTR)
		{
	   		DCU_CallbackPROGD_2( );
		}
		DCULITE.INT_STATUS.R = 0x00000400u;
    }
    if( DCULITE.INT_STATUS.B.DMA_TRANS_FINISH != 0u )
    {   
    	/* Rationale: It is required to assign a values to this function that will be used as callback */
		if (DCU_CallbackDMAD_2 != NULL_PTR)
		{
	   		DCU_CallbackDMAD_2( );
		}
		DCULITE.INT_STATUS.R = 0x00004000u;
    }       

}
#endif